import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Trophy, TrendingUp } from "lucide-react"

interface TopPerformersProps {
  performers: Array<{ name: string; deals: number; revenue: number }>
}

export function TopPerformers({ performers }: TopPerformersProps) {
  const sortedPerformers = [...performers].sort((a, b) => b.revenue - a.revenue).slice(0, 5)

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("es-AR", {
      style: "currency",
      currency: "ARS",
      minimumFractionDigits: 0,
    }).format(amount)
  }

  const getInitials = (name: string) => {
    return name
      .split(" ")
      .map((n) => n[0])
      .join("")
      .toUpperCase()
      .slice(0, 2)
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center gap-2">
          <Trophy className="h-5 w-5 text-amber-600" />
          <CardTitle>Top Performers</CardTitle>
        </div>
        <CardDescription>Los 5 mejores agentes del mes</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {sortedPerformers.map((performer, index) => (
            <div key={index} className="flex items-center justify-between p-4 bg-slate-50 rounded-lg">
              <div className="flex items-center gap-4">
                <div className="relative">
                  <Avatar className={index === 0 ? "ring-2 ring-amber-400" : ""}>
                    <AvatarFallback
                      className={
                        index === 0
                          ? "bg-amber-100 text-amber-700"
                          : index === 1
                            ? "bg-slate-200 text-slate-700"
                            : index === 2
                              ? "bg-orange-100 text-orange-700"
                              : "bg-blue-100 text-blue-700"
                      }
                    >
                      {getInitials(performer.name)}
                    </AvatarFallback>
                  </Avatar>
                  {index < 3 && (
                    <div
                      className={`absolute -top-1 -right-1 w-6 h-6 rounded-full flex items-center justify-center text-xs font-bold ${
                        index === 0
                          ? "bg-amber-400 text-white"
                          : index === 1
                            ? "bg-slate-400 text-white"
                            : "bg-orange-400 text-white"
                      }`}
                    >
                      {index + 1}
                    </div>
                  )}
                </div>
                <div>
                  <p className="font-semibold text-slate-900">{performer.name}</p>
                  <p className="text-sm text-slate-600">{performer.deals} ventas cerradas</p>
                </div>
              </div>
              <div className="text-right">
                <p className="font-bold text-emerald-600 text-lg">{formatCurrency(performer.revenue)}</p>
                <div className="flex items-center gap-1 text-xs text-emerald-600">
                  <TrendingUp className="h-3 w-3" />
                  <span>Total generado</span>
                </div>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  )
}
