"use client"

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { useMemo } from "react"

interface RevenueChartProps {
  deals: Array<{ amount: string; created_at: string; stage: string }>
}

export function RevenueChart({ deals }: RevenueChartProps) {
  const chartData = useMemo(() => {
    const last12Months = Array.from({ length: 12 }, (_, i) => {
      const date = new Date()
      date.setMonth(date.getMonth() - (11 - i))
      date.setDate(1)
      date.setHours(0, 0, 0, 0)
      return date
    })

    return last12Months.map((date) => {
      const nextMonth = new Date(date)
      nextMonth.setMonth(nextMonth.getMonth() + 1)

      const monthDeals = deals.filter((deal) => {
        const dealDate = new Date(deal.created_at)
        return dealDate >= date && dealDate < nextMonth
      })

      const wonDeals = monthDeals.filter((d) => d.stage === "closed-won")
      const revenue = wonDeals.reduce((sum, deal) => sum + Number(deal.amount), 0)

      return {
        month: date.toLocaleDateString("es-AR", { month: "short" }),
        revenue: revenue,
        deals: wonDeals.length,
      }
    })
  }, [deals])

  const maxRevenue = Math.max(...chartData.map((d) => d.revenue), 1)

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("es-AR", {
      style: "currency",
      currency: "ARS",
      minimumFractionDigits: 0,
      notation: "compact",
    }).format(amount)
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Ingresos - Últimos 12 Meses</CardTitle>
        <CardDescription>Ventas cerradas por mes</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {chartData.map((month, index) => (
            <div key={index} className="space-y-2">
              <div className="flex items-center justify-between text-sm">
                <span className="font-medium text-slate-700 capitalize">{month.month}</span>
                <div className="flex items-center gap-4">
                  <span className="text-slate-600">{month.deals} ventas</span>
                  <span className="font-semibold text-emerald-600">{formatCurrency(month.revenue)}</span>
                </div>
              </div>
              <div className="h-10 bg-slate-100 rounded-lg overflow-hidden">
                <div
                  className="h-full bg-gradient-to-r from-emerald-500 to-green-600 rounded-lg transition-all"
                  style={{ width: `${(month.revenue / maxRevenue) * 100}%` }}
                />
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  )
}
