"use client"

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { useMemo } from "react"

interface LeadsChartProps {
  leads: Array<{ created_at: string; status: string }>
}

export function LeadsChart({ leads }: LeadsChartProps) {
  const chartData = useMemo(() => {
    const last7Days = Array.from({ length: 7 }, (_, i) => {
      const date = new Date()
      date.setDate(date.getDate() - (6 - i))
      date.setHours(0, 0, 0, 0)
      return date
    })

    return last7Days.map((date) => {
      const nextDay = new Date(date)
      nextDay.setDate(nextDay.getDate() + 1)

      const dayLeads = leads.filter((lead) => {
        const leadDate = new Date(lead.created_at)
        return leadDate >= date && leadDate < nextDay
      })

      return {
        date: date.toLocaleDateString("es-AR", { weekday: "short", day: "numeric" }),
        total: dayLeads.length,
        new: dayLeads.filter((l) => l.status === "new").length,
        qualified: dayLeads.filter((l) => l.status === "qualified").length,
      }
    })
  }, [leads])

  const maxValue = Math.max(...chartData.map((d) => d.total), 1)

  return (
    <Card>
      <CardHeader>
        <CardTitle>Leads - Últimos 7 Días</CardTitle>
        <CardDescription>Nuevos leads por día</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {chartData.map((day, index) => (
            <div key={index} className="space-y-2">
              <div className="flex items-center justify-between text-sm">
                <span className="font-medium text-slate-700">{day.date}</span>
                <span className="text-slate-600">{day.total} leads</span>
              </div>
              <div className="flex gap-1 h-8">
                <div
                  className="bg-blue-500 rounded transition-all hover:bg-blue-600"
                  style={{ width: `${(day.new / maxValue) * 100}%`, minWidth: day.new > 0 ? "4%" : "0" }}
                  title={`Nuevos: ${day.new}`}
                />
                <div
                  className="bg-green-500 rounded transition-all hover:bg-green-600"
                  style={{ width: `${(day.qualified / maxValue) * 100}%`, minWidth: day.qualified > 0 ? "4%" : "0" }}
                  title={`Calificados: ${day.qualified}`}
                />
              </div>
            </div>
          ))}
        </div>
        <div className="flex items-center gap-4 mt-6 text-sm">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-blue-500 rounded" />
            <span>Nuevos</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-green-500 rounded" />
            <span>Calificados</span>
          </div>
        </div>
      </CardContent>
    </Card>
  )
}
