"use client"

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { useMemo } from "react"

interface CallsChartProps {
  calls: Array<{ created_at: string; call_status: string; outcome: string | null }>
}

export function CallsChart({ calls }: CallsChartProps) {
  const chartData = useMemo(() => {
    const last7Days = Array.from({ length: 7 }, (_, i) => {
      const date = new Date()
      date.setDate(date.getDate() - (6 - i))
      date.setHours(0, 0, 0, 0)
      return date
    })

    return last7Days.map((date) => {
      const nextDay = new Date(date)
      nextDay.setDate(nextDay.getDate() + 1)

      const dayCalls = calls.filter((call) => {
        const callDate = new Date(call.created_at)
        return callDate >= date && callDate < nextDay
      })

      return {
        date: date.toLocaleDateString("es-AR", { weekday: "short", day: "numeric" }),
        total: dayCalls.length,
        completed: dayCalls.filter((c) => c.call_status === "completed").length,
        interested: dayCalls.filter((c) => c.outcome === "interested").length,
      }
    })
  }, [calls])

  const maxValue = Math.max(...chartData.map((d) => d.total), 1)

  return (
    <Card>
      <CardHeader>
        <CardTitle>Llamadas - Últimos 7 Días</CardTitle>
        <CardDescription>Actividad de llamadas por día</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {chartData.map((day, index) => (
            <div key={index} className="space-y-2">
              <div className="flex items-center justify-between text-sm">
                <span className="font-medium text-slate-700">{day.date}</span>
                <span className="text-slate-600">{day.total} llamadas</span>
              </div>
              <div className="flex gap-1 h-8">
                <div
                  className="bg-purple-500 rounded transition-all hover:bg-purple-600"
                  style={{ width: `${(day.completed / maxValue) * 100}%`, minWidth: day.completed > 0 ? "4%" : "0" }}
                  title={`Completadas: ${day.completed}`}
                />
                <div
                  className="bg-emerald-500 rounded transition-all hover:bg-emerald-600"
                  style={{
                    width: `${(day.interested / maxValue) * 100}%`,
                    minWidth: day.interested > 0 ? "4%" : "0",
                  }}
                  title={`Interesados: ${day.interested}`}
                />
              </div>
            </div>
          ))}
        </div>
        <div className="flex items-center gap-4 mt-6 text-sm">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-purple-500 rounded" />
            <span>Completadas</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-emerald-500 rounded" />
            <span>Interesados</span>
          </div>
        </div>
      </CardContent>
    </Card>
  )
}
