"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { getSupabaseBrowserClient } from "@/lib/supabase/client"
import { Loader2 } from "lucide-react"
import { useToast } from "@/hooks/use-toast"

interface LogCallDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function LogCallDialog({ open, onOpenChange }: LogCallDialogProps) {
  const [loading, setLoading] = useState(false)
  const [leads, setLeads] = useState<any[]>([])
  const router = useRouter()
  const supabase = getSupabaseBrowserClient()
  const { toast } = useToast()

  const [formData, setFormData] = useState({
    lead_id: "",
    call_type: "outbound",
    call_status: "completed",
    duration_seconds: "",
    outcome: "",
    notes: "",
    scheduled_callback: "",
  })

  useEffect(() => {
    if (open) {
      loadLeads()
    }
  }, [open])

  const loadLeads = async () => {
    const { data } = await supabase
      .from("leads")
      .select("id, first_name, last_name, phone")
      .order("created_at", { ascending: false })

    if (data) setLeads(data)
  }

  const handleDurationChange = (minutes: string) => {
    const mins = Number.parseInt(minutes) || 0
    setFormData({ ...formData, duration_seconds: (mins * 60).toString() })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)

    try {
      const {
        data: { user },
      } = await supabase.auth.getUser()

      const { error } = await supabase.from("calls").insert({
        lead_id: formData.lead_id,
        agent_id: user?.id,
        call_type: formData.call_type,
        call_status: formData.call_status,
        duration_seconds: Number.parseInt(formData.duration_seconds) || 0,
        outcome: formData.outcome || null,
        notes: formData.notes,
        scheduled_callback: formData.scheduled_callback || null,
      })

      if (error) throw error

      // Update lead status based on outcome
      if (formData.outcome === "interested") {
        await supabase.from("leads").update({ status: "qualified" }).eq("id", formData.lead_id)
      } else if (formData.outcome === "callback") {
        await supabase.from("leads").update({ status: "contacted" }).eq("id", formData.lead_id)
      }

      toast({
        title: "Llamada registrada",
        description: "La llamada ha sido registrada exitosamente.",
      })

      onOpenChange(false)
      router.refresh()

      setFormData({
        lead_id: "",
        call_type: "outbound",
        call_status: "completed",
        duration_seconds: "",
        outcome: "",
        notes: "",
        scheduled_callback: "",
      })
    } catch (error: any) {
      toast({
        title: "Error",
        description: error.message || "No se pudo registrar la llamada",
        variant: "destructive",
      })
    } finally {
      setLoading(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>Registrar Llamada</DialogTitle>
          <DialogDescription>Registra los detalles de la llamada con el cliente</DialogDescription>
        </DialogHeader>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-2 gap-4 py-4">
            <div className="space-y-2 col-span-2">
              <Label htmlFor="lead_id">Cliente *</Label>
              <Select value={formData.lead_id} onValueChange={(value) => setFormData({ ...formData, lead_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar cliente" />
                </SelectTrigger>
                <SelectContent>
                  {leads.map((lead) => (
                    <SelectItem key={lead.id} value={lead.id}>
                      {lead.first_name} {lead.last_name} - {lead.phone}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="call_type">Tipo de Llamada</Label>
              <Select
                value={formData.call_type}
                onValueChange={(value) => setFormData({ ...formData, call_type: value })}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="outbound">Saliente</SelectItem>
                  <SelectItem value="inbound">Entrante</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="call_status">Estado</Label>
              <Select
                value={formData.call_status}
                onValueChange={(value) => setFormData({ ...formData, call_status: value })}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="completed">Completada</SelectItem>
                  <SelectItem value="no-answer">Sin respuesta</SelectItem>
                  <SelectItem value="busy">Ocupado</SelectItem>
                  <SelectItem value="voicemail">Buzón de voz</SelectItem>
                  <SelectItem value="failed">Fallida</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="duration">Duración (minutos)</Label>
              <Input
                id="duration"
                type="number"
                placeholder="5"
                onChange={(e) => handleDurationChange(e.target.value)}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="outcome">Resultado</Label>
              <Select value={formData.outcome} onValueChange={(value) => setFormData({ ...formData, outcome: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="Seleccionar resultado" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="interested">Interesado</SelectItem>
                  <SelectItem value="callback">Callback</SelectItem>
                  <SelectItem value="not-interested">No interesado</SelectItem>
                  <SelectItem value="wrong-number">Número erróneo</SelectItem>
                  <SelectItem value="appointment-set">Cita agendada</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {formData.outcome === "callback" && (
              <div className="space-y-2 col-span-2">
                <Label htmlFor="scheduled_callback">Fecha y Hora de Callback</Label>
                <Input
                  id="scheduled_callback"
                  type="datetime-local"
                  value={formData.scheduled_callback}
                  onChange={(e) => setFormData({ ...formData, scheduled_callback: e.target.value })}
                />
              </div>
            )}

            <div className="space-y-2 col-span-2">
              <Label htmlFor="notes">Notas de la Llamada</Label>
              <Textarea
                id="notes"
                placeholder="Detalles de la conversación..."
                value={formData.notes}
                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                rows={4}
              />
            </div>
          </div>

          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)} disabled={loading}>
              Cancelar
            </Button>
            <Button type="submit" disabled={loading}>
              {loading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
              Registrar Llamada
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
