"use client"

import { useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Card } from "@/components/ui/card"
import { Search, Phone, PhoneIncoming, Clock, Calendar } from "lucide-react"
import { format } from "date-fns"
import { es } from "date-fns/locale"

interface CallsTableProps {
  calls: any[]
}

export function CallsTable({ calls }: CallsTableProps) {
  const [searchTerm, setSearchTerm] = useState("")

  const filteredCalls = calls.filter(
    (call) =>
      call.lead?.first_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      call.lead?.last_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      call.lead?.phone?.includes(searchTerm) ||
      call.agent?.full_name?.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  const getCallTypeBadge = (type: string) => {
    return type === "outbound" ? (
      <Badge className="bg-blue-100 text-blue-700">
        <Phone className="h-3 w-3 mr-1" />
        Saliente
      </Badge>
    ) : (
      <Badge className="bg-green-100 text-green-700">
        <PhoneIncoming className="h-3 w-3 mr-1" />
        Entrante
      </Badge>
    )
  }

  const getCallStatusBadge = (status: string) => {
    const statusConfig: Record<string, { label: string; className: string }> = {
      completed: { label: "Completada", className: "bg-green-100 text-green-700" },
      "no-answer": { label: "Sin respuesta", className: "bg-yellow-100 text-yellow-700" },
      busy: { label: "Ocupado", className: "bg-orange-100 text-orange-700" },
      voicemail: { label: "Buzón de voz", className: "bg-blue-100 text-blue-700" },
      failed: { label: "Fallida", className: "bg-red-100 text-red-700" },
    }

    const config = statusConfig[status] || { label: status, className: "bg-slate-100 text-slate-700" }
    return <Badge className={config.className}>{config.label}</Badge>
  }

  const getOutcomeBadge = (outcome: string | null) => {
    if (!outcome) return null

    const outcomeConfig: Record<string, { label: string; className: string }> = {
      interested: { label: "Interesado", className: "bg-emerald-100 text-emerald-700" },
      callback: { label: "Callback", className: "bg-purple-100 text-purple-700" },
      "not-interested": { label: "No interesado", className: "bg-red-100 text-red-700" },
      "wrong-number": { label: "Número erróneo", className: "bg-slate-100 text-slate-700" },
      "appointment-set": { label: "Cita agendada", className: "bg-blue-100 text-blue-700" },
    }

    const config = outcomeConfig[outcome] || { label: outcome, className: "bg-slate-100 text-slate-700" }
    return <Badge className={config.className}>{config.label}</Badge>
  }

  const formatDuration = (seconds: number) => {
    const minutes = Math.floor(seconds / 60)
    const secs = seconds % 60
    return `${minutes}:${secs.toString().padStart(2, "0")}`
  }

  return (
    <Card>
      <div className="p-4 border-b border-slate-200">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-slate-400" />
          <Input
            placeholder="Buscar por cliente, agente o teléfono..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="pl-10"
          />
        </div>
      </div>

      <div className="overflow-x-auto">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Fecha/Hora</TableHead>
              <TableHead>Cliente</TableHead>
              <TableHead>Tipo</TableHead>
              <TableHead>Estado</TableHead>
              <TableHead>Resultado</TableHead>
              <TableHead>Duración</TableHead>
              <TableHead>Agente</TableHead>
              <TableHead>Callback</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredCalls.length === 0 ? (
              <TableRow>
                <TableCell colSpan={8} className="text-center py-12 text-slate-500">
                  No se encontraron llamadas
                </TableCell>
              </TableRow>
            ) : (
              filteredCalls.map((call) => (
                <TableRow key={call.id}>
                  <TableCell>
                    <div className="flex flex-col">
                      <span className="text-sm font-medium">
                        {format(new Date(call.created_at), "dd/MM/yyyy", { locale: es })}
                      </span>
                      <span className="text-xs text-slate-500">
                        {format(new Date(call.created_at), "HH:mm", { locale: es })}
                      </span>
                    </div>
                  </TableCell>
                  <TableCell>
                    {call.lead ? (
                      <div>
                        <p className="font-medium text-slate-900">
                          {call.lead.first_name} {call.lead.last_name}
                        </p>
                        <p className="text-xs text-slate-500">{call.lead.phone}</p>
                      </div>
                    ) : (
                      <span className="text-slate-400">Sin cliente</span>
                    )}
                  </TableCell>
                  <TableCell>{getCallTypeBadge(call.call_type)}</TableCell>
                  <TableCell>{getCallStatusBadge(call.call_status)}</TableCell>
                  <TableCell>{getOutcomeBadge(call.outcome)}</TableCell>
                  <TableCell>
                    <div className="flex items-center gap-2">
                      <Clock className="h-3 w-3 text-slate-400" />
                      <span className="text-sm">{formatDuration(call.duration_seconds || 0)}</span>
                    </div>
                  </TableCell>
                  <TableCell>
                    <span className="text-sm">{call.agent?.full_name || "Sin asignar"}</span>
                  </TableCell>
                  <TableCell>
                    {call.scheduled_callback ? (
                      <div className="flex items-center gap-2 text-sm">
                        <Calendar className="h-3 w-3 text-purple-600" />
                        <span>{format(new Date(call.scheduled_callback), "dd/MM HH:mm", { locale: es })}</span>
                      </div>
                    ) : (
                      <span className="text-slate-400 text-sm">-</span>
                    )}
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>
    </Card>
  )
}
