import { requireAuth } from "@/lib/auth/get-user"
import { getSupabaseServerClient } from "@/lib/supabase/server"
import { CallsTable } from "@/components/calls/calls-table"
import { CallsHeader } from "@/components/calls/calls-header"
import { CallsStats } from "@/components/calls/calls-stats"

export default async function CallsPage() {
  const user = await requireAuth()
  const supabase = await getSupabaseServerClient()

  // Get calls based on user role
  let callsQuery = supabase
    .from("calls")
    .select(
      `
      *,
      lead:leads!calls_lead_id_fkey(id, first_name, last_name, phone, status),
      agent:users!calls_agent_id_fkey(full_name)
    `,
    )
    .order("created_at", { ascending: false })

  if (user.role === "agent") {
    callsQuery = callsQuery.eq("agent_id", user.id)
  }

  const { data: calls } = await callsQuery

  // Get today's calls count
  const today = new Date()
  today.setHours(0, 0, 0, 0)

  const todayCalls = calls?.filter((call) => new Date(call.created_at) >= today) || []

  // Calculate stats
  const totalCalls = calls?.length || 0
  const completedCalls = calls?.filter((c) => c.call_status === "completed").length || 0
  const totalDuration = calls?.reduce((sum, call) => sum + (call.duration_seconds || 0), 0) || 0
  const avgDuration = completedCalls > 0 ? Math.round(totalDuration / completedCalls) : 0

  const interestedCalls = calls?.filter((c) => c.outcome === "interested").length || 0
  const callbackScheduled = calls?.filter((c) => c.outcome === "callback").length || 0

  return (
    <div className="space-y-6">
      <CallsHeader />
      <CallsStats
        totalCalls={totalCalls}
        todayCalls={todayCalls.length}
        completedCalls={completedCalls}
        avgDuration={avgDuration}
        interestedCalls={interestedCalls}
        callbackScheduled={callbackScheduled}
      />
      <CallsTable calls={calls || []} />
    </div>
  )
}
